﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using BMS.VistaIntegration.Data.WF;
using BMS.VistaIntegration.Via.Commands;
using BMS.VistaIntegration.Via.Commands.EIS;
using BMS.VistaIntegration.Via.Commands.EVS;
using BMS.VistaIntegration.Via.Commands.WF;
using BMS.VistaIntegration.VistA;

// ReSharper disable InconsistentNaming

namespace BMS.VistaIntegration.Via
{
    public class ViaVistAQuery : IVistAQuery
    {
        private const int DEFAULT_BULK_COUNT = 1000;
        private readonly BmsLogger logger;

        private readonly ViaVistASession session;

        private readonly ListHospitalLocationCommand hospitalLocationCommand;
        private readonly ListSpecialtyCommand specialtyCommand;
        private readonly ListFacilityTreatingSpecialtyCommand facilityTreatingSpecialtyCommand;
        private readonly ListWardLocationCommand wardLocationCommand;
        private readonly ListFacilityMovementTypeCommand facilityMovementTypeCommand;
        private readonly ListMedicalCenterDivisionCommand medicalCenterDivisionCommand;
        private readonly ListOrderableItemCommand orderableItemCommand;
        private readonly ListRoomBedCommand roomBedCommand;
        private readonly GetPatientCommand getPatientCommand;
        private readonly ListPatientCommand listPatientCommand;
        private readonly ListAdmittedPatientsForUpdateCommand listAdmittedPatientsForUpdateCommand;
        private readonly ListNewPersonCommand listNewPersonCommand;
        private readonly ListPatientMovementCommand listPatientMovementCommand;
        private readonly ListScheduledAdmissionCommand listScheduledAdmissionCommand;
        private readonly ListPatientAppointmentsFromClinicCommand listPatientAppointmentsFromClinicCommand;
        private readonly GetPatientMovementCommand getPatientMovementCommand;
        private readonly ListOrderActionsCommand listOrderActionsCommand;
        private readonly ListOrderStatusesCommand listOrderStatusesCommand;
        private readonly ListPatientMovementIensCommand listPatientMovementIensCommand;
        private readonly ListAdmittedPatientsCommand listAdmittedPatientsCommand;
        private readonly ListBedSwitchCommand listBedSwitchCommand;
        private readonly ListCancelOrdersCommand listCanceledOrdersCommand;

        public ViaVistAQuery(ViaVistASession session)
        {
            this.session = session;
            this.logger = new BmsLogger(string.Format("ViaVistAQuery:{0} - ", this.Site.Name));
            BulkCount = DEFAULT_BULK_COUNT;

            var entitySetCache = this.EntitySetCache = EntitySetCache.Create(session.VistASite);
            // Register commands for types that are retrieve multiple times (as sub-entities).
            entitySetCache.RegisterBulkType<WardLocation>(WardLocationDependencySource.Instance);
            entitySetCache.RegisterBulkType<Patient>(PatientDependencySource.Instance);
            entitySetCache.RegisterBulkType<FacilityTreatingSpecialty>();
            entitySetCache.RegisterBulkType<NewPerson>();
            entitySetCache.RegisterBulkType<Specialty>();
            entitySetCache.RegisterBulkType<HospitalLocation>(HospitalLocationDependencySource.Instance);
            entitySetCache.RegisterBulkType<FacilityMovementType>();
            entitySetCache.RegisterBulkType<RoomBed>(RoomBedDependencySource.Instance);
            entitySetCache.RegisterBulkType<MedicalCenterDivision>();
            entitySetCache.RegisterBulkType<OrderStatus>();

            this.hospitalLocationCommand = new ListHospitalLocationCommand(this);
            this.specialtyCommand = new ListSpecialtyCommand(this);
            this.facilityTreatingSpecialtyCommand = new ListFacilityTreatingSpecialtyCommand(this);
            this.wardLocationCommand = new ListWardLocationCommand(this);
            this.facilityMovementTypeCommand = new ListFacilityMovementTypeCommand(this);
            this.medicalCenterDivisionCommand = new ListMedicalCenterDivisionCommand(this);
            this.orderableItemCommand = new ListOrderableItemCommand(this);
            this.roomBedCommand = new ListRoomBedCommand(this);
            this.getPatientCommand = new GetPatientCommand(this);
            this.listPatientCommand = new ListPatientCommand(this);
            this.listAdmittedPatientsForUpdateCommand = new ListAdmittedPatientsForUpdateCommand(this);
            this.listNewPersonCommand = new ListNewPersonCommand(this);
            this.listPatientMovementCommand = new ListPatientMovementCommand(this);
            this.listScheduledAdmissionCommand = new ListScheduledAdmissionCommand(this);
            this.listPatientAppointmentsFromClinicCommand = new ListPatientAppointmentsFromClinicCommand(this);
            this.getPatientMovementCommand = new GetPatientMovementCommand(this);
            this.listOrderActionsCommand = new ListOrderActionsCommand(this);
            this.listOrderStatusesCommand = new ListOrderStatusesCommand(this);
            this.listPatientMovementIensCommand = new ListPatientMovementIensCommand(this);
            this.listAdmittedPatientsCommand = new ListAdmittedPatientsCommand(this);
            this.listBedSwitchCommand = new ListBedSwitchCommand(this);
            this.listCanceledOrdersCommand = new ListCancelOrdersCommand(this);
        }

        public int BulkCount { get; set; }

        internal VistASite Site
        {
            get
            {
                return this.session.VistASite;
            }
        }

        public EntitySetCache EntitySetCache
        {
            get;
            private set;
        }

        public IList<HospitalLocation> GetHospitalLocations()
        {
            return GetResults(hospitalLocationCommand);
        }

        public IList<Patient> GetPatients(DateTime? startDate = null, DateTime? endDate = null, string vistaPatientIen = null)
        {
            this.listPatientCommand.StartDate = startDate;
            this.listPatientCommand.EndDate = endDate;
            this.listPatientCommand.VistaPatientIen = vistaPatientIen;

            return this.GetResults(this.listPatientCommand);
        }

        public IList<Patient> GetPatients(IEnumerable<string> ssns)
        {
            throw new NotImplementedException();
        }

        public IList<RoomBed> GetRoomBeds()
        {
            return this.GetResults(this.roomBedCommand);
        }

        public IList<WardLocation> GetWardLocations()
        {
            return this.GetResults(this.wardLocationCommand);
        }

        public IList<NewPerson> GetNewPersons(DateTime? startDate = null, DateTime? endDate = null)
        {
            this.listNewPersonCommand.StartDate = startDate;
            this.listNewPersonCommand.EndDate = endDate;

            return this.GetResults(this.listNewPersonCommand);
        }

        public IList<Patient> GetAdmittedPatientsForUpdate()
        {
            return this.GetResults(this.listAdmittedPatientsForUpdateCommand);
        }

        public IList<FacilityTreatingSpecialty> GetFacilityTreatingSpecialties()
        {
            IList<FacilityTreatingSpecialty> result = GetResults(facilityTreatingSpecialtyCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        public IList<FacilityMovementType> GetFacilityMovementTypes()
        {
            IList<FacilityMovementType> result = GetResults(facilityMovementTypeCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        public IList<OrderableItem> GetOrderableItems()
        {
            var result = GetResults(orderableItemCommand);

            return result == null ? null : result.Where(item => !string.IsNullOrWhiteSpace(item.Name)).ToList();
        }

        public IList<Specialty> GetSpecialties()
        {
            return this.GetResults(this.specialtyCommand);
        }

        public IList<OrderStatus> GetOrderStatuses()
        {
            return this.GetResults(this.listOrderStatusesCommand);
        }

        public IList<MedicalCenterDivision> GetMedicalCenterDivisions()
        {
            IList<MedicalCenterDivision> result = GetResults(medicalCenterDivisionCommand);
            if (result != null)
                return result.Where(a => a.Name != null && !string.IsNullOrEmpty(a.Name.Trim())).ToList();
            else
                return result;
        }

        public IList<OrderAction> GetOrderActions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> orderableItemsIen = null)
        {
            this.listOrderActionsCommand.StartDate = startDate;
            this.listOrderActionsCommand.EndDate = endDate;

            this.listOrderActionsCommand.PatientIen = patientIen;
            this.listOrderActionsCommand.OrderableItemIens = orderableItemsIen;

            return this.GetResults(this.listOrderActionsCommand);
        }

        public IList<PatientAppointment> GetPatientAppointments(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null, IEnumerable<string> clinicIens = null)
        {
            var clinicPatients = new Dictionary<string, IEnumerable<Patient>>();

            if (!string.IsNullOrEmpty(patientIen))
            {
                EntitySetCache.DownloadEntities(typeof(Patient), this, this.session.VistASite, new[] { patientIen }, '\0', ListPatientCommand.Target, ListPatientCommand.ArgumentsCount);
                var patient = EntitySetCache.GetEntity<Patient>(this, patientIen);

                clinicPatients.Add(string.Empty, new Patient[] { patient });
            }

            if (clinicIens != null)
            {
                foreach (var clinicIen in clinicIens)
                {
                    var listClinicAppointmentsCommand = new ListClinicAppointmentsCommand(this)
                    {
                        StartDate = startDate,
                        EndDate = endDate,
                        ClinicIen = clinicIen,
                    };

                    var clinicAppointments = this.GetResults(listClinicAppointmentsCommand);
                    var patientIens = (from appointment in clinicAppointments
                                       from ien in appointment.PatientIens
                                       select ien).Distinct().ToArray();

                    EntitySetCache.DownloadEntities(typeof(Patient), this, this.session.VistASite, patientIens, '\0', ListPatientCommand.Target, ListPatientCommand.ArgumentsCount);
                    var patients = EntitySetCache.GetEntities<Patient>(this, patientIens).ToArray();
                    clinicPatients.Add(clinicIen, patients);
                }
            }

            this.listPatientAppointmentsFromClinicCommand.StartDate = startDate;
            this.listPatientAppointmentsFromClinicCommand.EndDate = endDate;
            this.listPatientAppointmentsFromClinicCommand.ClinicPatients = clinicPatients;

            return this.GetResults(this.listPatientAppointmentsFromClinicCommand);
        }

        public IList<PatientMovement> GetPatientMovements(DateTime? startDate = null, DateTime? endDate = null)
        {
            this.listPatientMovementCommand.StartDate = startDate;
            this.listPatientMovementCommand.EndDate = endDate;

            return this.GetResults(this.listPatientMovementCommand);
        }

        public IList<PatientMovement> GetPatientMovementsForAdmission(string admissionIen)
        {
            throw new NotImplementedException();
        }

        public IList<AdmittedPatient> GetAdmittedPatients(string lastPatientMovementIen)
        {
            this.listAdmittedPatientsCommand.MovementIen = lastPatientMovementIen;

            return this.GetResults(this.listAdmittedPatientsCommand).Where(x => !string.IsNullOrEmpty(x.MovementIen)).ToList();
        }

        public IList<ScheduledAdmission> GetScheduledAdmissions(DateTime? startDate = null, DateTime? endDate = null, string patientIen = null)
        {
            this.listScheduledAdmissionCommand.StartDate = startDate;
            this.listScheduledAdmissionCommand.EndDate = endDate;
            this.listScheduledAdmissionCommand.PatientIen = patientIen;

            return this.GetResults(this.listScheduledAdmissionCommand);
        }

        public PatientMovement GetPatientMovement(string patientIen, DateTime dateTime,
            MovementTransactionType movementType)
        {
            throw new NotImplementedException();
        }

        public PatientMovement GetPatientMovementByIen(string patientMovementIen)
        {
            return this.GetResult(this.getPatientMovementCommand, patientMovementIen);
        }

        public IList<PatientMovementIen> GetPatientMovementIens(DateTime startDate, DateTime endDate)
        {
            this.listPatientMovementIensCommand.StartDate = startDate;
            this.listPatientMovementIensCommand.EndDate = endDate;

            return this.GetResults(this.listPatientMovementIensCommand);
        }

        public Patient GetPatientByIen(string ien)
        {
            try
            {
                return this.GetResult(getPatientCommand, ien);
            }
            catch (ViaException ex)
            {
                this.logger.LogFormat(BmsLogger.Level.Error, "Exception: {0}", ex);

                return null;
            }
        }

        public IList<BedSwitch> GetBedsSwitch(IEnumerable<string> iens)
        {
            this.listBedSwitchCommand.Iens = iens.ToArray();

            return this.GetResults(this.listBedSwitchCommand);
        }

        public IList<String> GetCanceledOrders(IEnumerable<string> iens)
        {
            return (from ien in iens
                    let cancelOrder = this.GetResult(this.listCanceledOrdersCommand, ien)
                    where cancelOrder != null && cancelOrder.StatusId != "6"
                    select cancelOrder.IEN).ToArray();
        }

        internal List<T> GetResults<T>(BaseListCommand<T> command) where T : class, IEntity, new()
        {
            List<T> result = null;

            if (command.HasMore)
            {
                if (!string.IsNullOrEmpty(command.Last))
                    session.RunClientAction(() => session.IsAlive());
                command.MaxCount = BulkCount;

                try
                {
                    session.RunClientAction(() => result = command.Execute(session));
                }
                catch (Exception ex)
                {
                    this.logger.LogFormat(BmsLogger.Level.Error, "GetResults - exception\r\n{0}", ex);
                    throw;
                }
            }

            if (result == null)
            {
                this.logger.LogFormat(BmsLogger.Level.Verbose, "{0}: Returning no records.", command.GetType().GetFriendlyName());
            }
            else
            {
                this.logger.LogFormat(BmsLogger.Level.Verbose, "{0}: Returning {1} record(s).", command.GetType().GetFriendlyName(), result.Count);
            }

            return result;
        }

        private T GetResult<T>(BaseSingleCommand<T> command, string ien) where T : class, IEntity, new()
        {
            command.Ien = ien;

            try
            {
                T result = null;
                session.RunClientAction(() =>
                {
                    var results = command.Execute(session);
                    if (results != null)
                    {
                        result = results.SingleOrDefault();
                    }
                });
                return result;
            }
            catch (Exception ex)
            {
                this.logger.LogFormat(BmsLogger.Level.Error, "GetResult - exception\r\n{0}", ex);
                throw;
            }
        }
    }
}
